/*
   This file shows how to create a child process that has its
   standard input and standard output streams redirected to files.
*/
#include <stdio.h>
#include <windows.h>

/* function prototype */
void printError(char* functionName);

int main( int argc, char *argv[])
{
   char  inputName[] = "double.c"; /* should make these command line arguments */
   char outputName[] = "test.txt";

   LPTSTR lpCommandLine;
   PROCESS_INFORMATION processInfo;
   HANDLE hIn;
   HANDLE hOut;

   STARTUPINFO startInfo;
   ZeroMemory(&startInfo, sizeof(startInfo));
   startInfo.cb = sizeof(startInfo);

   hIn = CreateFile(inputName,             // name of the file to read
                    GENERIC_READ,          // open for reading
                    0,                     // do not share
                    NULL,                  // default security
                    OPEN_EXISTING,         // open file only if it exists
                    FILE_ATTRIBUTE_NORMAL, // normal file
                    NULL);                 // no attr. template
   if (hIn == INVALID_HANDLE_VALUE)
   {
      char buf[100];
      sprintf(buf, "CreateFile(%s)", inputName);
      printError( buf );
   }
   /* to use this handle for stdin, the handle must be inheritable */
   if ( ! SetHandleInformation(hIn, HANDLE_FLAG_INHERIT, HANDLE_FLAG_INHERIT) )
   {
      printError("SetHandleInformation(hIn)");
   }

   hOut = CreateFile(outputName,            // name of the file to write
                     GENERIC_WRITE,         // open for writing
                     0,                     // do not share
                     NULL,                  // default security
                     CREATE_ALWAYS,         // overwrite an existing file
                     FILE_ATTRIBUTE_NORMAL, // normal file
                     NULL);                 // no attr. template
   if (hOut == INVALID_HANDLE_VALUE)
   {
      char buf[100];
      sprintf(buf, "CreateFile(%s)", outputName);
      printError( buf );
   }
   /* to use this handle for stdout, the handle must be inheritable */
   if ( ! SetHandleInformation(hOut, HANDLE_FLAG_INHERIT, HANDLE_FLAG_INHERIT) )
   {
      printError("SetHandleInformation(hOut)");
   }

   /* redirect the child's standard in and out to the files */
   /* start with the inherited handlles */
   startInfo.dwFlags |= STARTF_USESTDHANDLES;
   startInfo.hStdInput  = GetStdHandle(STD_INPUT_HANDLE);
   startInfo.hStdOutput = GetStdHandle(STD_OUTPUT_HANDLE);
   startInfo.hStdError  = GetStdHandle(STD_ERROR_HANDLE);
   /* now redirect stdin and stdout to the files */
   startInfo.hStdInput = hIn;
   startInfo.hStdOutput = hOut;

   /* set up the command line */
   lpCommandLine = "double.exe";

   /* create the child process */
   if( !CreateProcess(NULL, lpCommandLine, NULL, NULL, TRUE,
                      NORMAL_PRIORITY_CLASS,
                      NULL, NULL, &startInfo, &processInfo) )
   {
      char buf[100];
      sprintf(buf, "CreateProcess(%s)", lpCommandLine);
      printError( buf );
   }
   else
   {
      fprintf(stderr, "Started process %d\n", (int)processInfo.dwProcessId);
   }

   /* wait for the child process to end */
   WaitForSingleObject(processInfo.hProcess, INFINITE);

   /* close all the handles */
   CloseHandle(processInfo.hProcess);
   CloseHandle(processInfo.hThread);
   CloseHandle(hIn);
   CloseHandle(hOut);

   return 0;
}



/****************************************************************
   The following function can be used to print out "meaningful"
   error messages. If you call a Win32 function and it returns
   with an error condition, then call this function right away and
   pass it a string containing the name of the Win32 function that
   failed. This function will print out a reasonable text message
   explaining the error and then (if chosen) terminate the program.
*/
void printError(char* functionName)
{
   LPVOID lpMsgBuf;
    int error_no;
    error_no = GetLastError();
    FormatMessage(
         FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
         NULL,
         error_no,
         MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
         (LPTSTR) &lpMsgBuf,
         0,
         NULL
    );
    // Display the string.
    fprintf(stderr, "\n%s failed on error %d: ", functionName, error_no);
    fprintf(stderr, (const char*)lpMsgBuf);
    // Free the buffer.
    LocalFree( lpMsgBuf );
    //ExitProcess(1);  // terminate the program
}//printError
